// archivo js/appHabAca.js

// Función para obtener datos desde un endpoint PHP con manejo de errores
async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) {
      throw new Error(`Error en la petición HTTP: ${response.status}`);
    }
    return await response.json();
  } catch (error) {
    console.error(`Fetch error en ${url}:`, error);
    return null;
  }
}

// Inicialización de la gráfica de evolución (línea) con múltiples datasets
async function initEvolutionChart() {
  // Asegúrate de que la ruta sea correcta (si tus PHP están en la carpeta "php")
  const evolutionData = await fetchData('get_evolutionHabAca_data.php');
  if (!evolutionData) {
    console.error("No se pudieron obtener datos para la gráfica de evolución.");
    return;
  }
  
  // Extraer etiquetas (meses)
  const labels = evolutionData.map(item => item.mes);
  
  // Extraer aciertos de cada test
  const aciertosLecturaComprension = evolutionData.map(item => item.lectura_comprensión.total_aciertos);
  const aciertosEscritura = evolutionData.map(item => item.escritura.total_aciertos);
  const aciertosMatematicasBasicas = evolutionData.map(item => item.matematicas_basicas.total_aciertos);


  // Verificar que exista el canvas con id "evolutionChart"
  const canvasLine = document.getElementById('evolutionChart');
  if (!canvasLine) {
    console.error('No se encontró el elemento canvas con id "evolutionChart".');
    return;
  }
  
  const ctxLine = canvasLine.getContext('2d');
  new Chart(ctxLine, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Evolución - Comprensión Lectora',
          data: aciertosLecturaComprension,
          borderColor: 'rgba(75, 192, 192, 1)',
          backgroundColor: 'rgba(75, 192, 192, 0.2)',
          fill: true,
        },
        {
          label: 'Evolución - Escritura',
          data: aciertosEscritura,
          borderColor: 'rgba(255, 99, 132, 1)',
          backgroundColor: 'rgba(255, 99, 132, 0.2)',
          fill: true,
        },
        {
          label: 'Evolución - Matemáticas Básicas',
          data: aciertosMatematicasBasicas,
          borderColor: 'rgba(255, 206, 86, 1)',
          backgroundColor: 'rgba(255, 206, 86, 0.2)',
          fill: true,
        }
      ]
    },
    options: {
      responsive: true,
      plugins: {
        zoom: {
          zoom: {
            wheel: { enabled: true },
            pinch: { enabled: true },
            mode: 'x'
          },
          pan: { enabled: true, mode: 'x' }
        }
      },
      scales: {
        x: { title: { display: true, text: 'Mes/Año' } },
        y: { title: { display: true, text: 'Puntaje' } }
      }
    }
  });
}

// Inicialización de la gráfica de torta (pie chart) con porcentajes en tooltips y en labels
async function initPieChart() {
  const pieData = await fetchData('get_pieHabAca_data.php');
  if (!pieData) {
    console.error("No se pudieron obtener datos para la gráfica de torta.");
    return;
  }
  
  const pieLabels = pieData.map(item => item.test);
  const pieValues = pieData.map(item => item.total_aciertos);
  
  // Calcular el total de aciertos para usarlo en el cálculo de porcentajes
  const totalAciertos = pieValues.reduce((acc, val) => acc + val, 0);
  
  const canvasPie = document.getElementById('pieChart');
  if (!canvasPie) {
    console.error('No se encontró el elemento canvas con id "pieChart".');
    return;
  }
  
  const ctxPie = canvasPie.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: [
          'rgba(75, 192, 192, 0.6)',
          'rgba(255, 99, 132, 0.6)',
          'rgba(255, 206, 86, 0.6)'
        ]
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const percentage = totalAciertos > 0 ? ((value / totalAciertos) * 100).toFixed(2) : 0;
              return `${label}: ${percentage}% (${value})`;
            }
          }
        },
        datalabels: {
          color: '#fff',
          formatter: function(value) {
            const percentage = totalAciertos > 0 ? ((value / totalAciertos) * 100).toFixed(2) : 0;
            return percentage + '%';
          }
        }
      }
    },
    plugins: [ChartDataLabels] // Incluir el plugin de datalabels
  });
}

// Inicializar las gráficas al cargar la página
window.onload = function() {
  initEvolutionChart();
  initPieChart();
};
