// Archivo: js/appInd.js

// Extraer el ID del usuario del campo oculto (asegúrate de que exista un elemento con id "participant_id")
const userId = document.getElementById('participant_id') ? document.getElementById('participant_id').value : '';

const colorPalette = [
  { border: 'rgba(75, 192, 192, 1)', background: 'rgba(75, 192, 192, 0.2)' },
  { border: 'rgba(255, 99, 132, 1)', background: 'rgba(255, 99, 132, 0.2)' },
  { border: 'rgba(54, 162, 235, 1)', background: 'rgba(54, 162, 235, 0.2)' },
  { border: 'rgba(255, 206, 86, 1)', background: 'rgba(255, 206, 86, 0.2)' }
];

// Función para obtener datos desde un endpoint PHP con manejo de errores
async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) {
      throw new Error(`Error en la petición HTTP: ${response.status}`);
    }
    return await response.json();
  } catch (error) {
    console.error(`Fetch error en ${url}:`, error);
    return null;
  }
}

// Inicialización de la gráfica de evolución (línea) con múltiples datasets
async function initEvolutionChart() {
  const evolutionData = await fetchData('get_evolutionInd_data.php?id=' + userId);
  if (!evolutionData) {
    console.error("No se pudieron obtener datos para la gráfica de evolución.");
    return;
  }
  
  const labels = evolutionData.map(item => item.mes);
  
  // Extraer los aciertos para cada test; se asume que la respuesta tiene las claves correspondientes
  const aciertosMemoriaVerbal = evolutionData.map(item => item.test_entries ? item.test_entries.total_aciertos : 0);
  const aciertosFuncionesEjecutivas = evolutionData.map(item => item.funciones_ejecutivas ? item.funciones_ejecutivas.total_aciertos : 0);
  const aciertosAtencionVelocidad = evolutionData.map(item => item.atencion_velocidad_procesamiento ? item.atencion_velocidad_procesamiento.total_aciertos : 0);
  const aciertosVisuoespacial = evolutionData.map(item => item.visuoespacial_results ? item.visuoespacial_results.total_aciertos : 0);

  const canvasLine = document.getElementById('evolutionChart');
  if (!canvasLine) {
    console.error('No se encontró el elemento canvas con id "evolutionChart".');
    return;
  }
  
  const ctxLine = canvasLine.getContext('2d');
  new Chart(ctxLine, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Evolución - Memoria Verbal',
          data: aciertosMemoriaVerbal,
          borderColor: colorPalette[0].border,
          backgroundColor: colorPalette[0].background,
          fill: true,
        },
        {
          label: 'Evolución - Funciones Ejecutivas',
          data: aciertosFuncionesEjecutivas,
          borderColor: colorPalette[1].border,
          backgroundColor: colorPalette[1].background,
          fill: true,
        },
        {
          label: 'Evolución - Atención y Velocidad de Procesamiento',
          data: aciertosAtencionVelocidad,
          borderColor: colorPalette[2].border,
          backgroundColor: colorPalette[2].background,
          fill: true,
        },
        {
          label: 'Evolución - Visuoespacial',
          data: aciertosVisuoespacial,
          borderColor: colorPalette[3].border,
          backgroundColor: colorPalette[3].background,
          fill: true,
        }
      ]
    },
    options: {
      responsive: true,
      plugins: {
        zoom: {
          zoom: {
            wheel: { enabled: true },
            pinch: { enabled: true },
            mode: 'x'
          },
          pan: { enabled: true, mode: 'x' }
        }
      },
      scales: {
        x: { title: { display: true, text: 'Mes/Año' } },
        y: { title: { display: true, text: 'Puntaje' } }
      }
    }
  });
}

// Inicialización de la gráfica de torta (pie chart) con porcentajes en tooltips y labels
async function initPieChart() {
  const pieData = await fetchData('get_pieInd_data.php?id=' + userId);
  if (!pieData) {
    console.error("No se pudieron obtener datos para la gráfica de torta.");
    return;
  }
  
  const pieLabels = pieData.map(item => item.test);
  const pieValues = pieData.map(item => item.total_aciertos);
  const totalAciertos = pieValues.reduce((acc, val) => acc + val, 0);
  
  const canvasPie = document.getElementById('pieChart');
  if (!canvasPie) {
    console.error('No se encontró el elemento canvas con id "pieChart".');
    return;
  }
  
  const ctxPie = canvasPie.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: colorPalette.map(c => c.border)
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const percentage = totalAciertos > 0 ? ((value / totalAciertos) * 100).toFixed(2) : 0;
              return `${label}: ${percentage}% (${value})`;
            }
          }
        },
        datalabels: {
          color: '#fff',
          formatter: function(value) {
            const percentage = totalAciertos > 0 ? ((value / totalAciertos) * 100).toFixed(2) : 0;
            return percentage + '%';
          }
        }
      }
    },
    plugins: [ChartDataLabels]
  });
}

window.onload = function() {
  initEvolutionChart();
  initPieChart();
};
