// archivo appIndHabAcaBas.js

// Extraer el ID del usuario del campo oculto
const userId = document.getElementById('participant_id').value;

const colorPalette = [
  { border: 'rgba(75, 192, 192, 1)', background: 'rgba(75, 192, 192, 0.2)' },
  { border: 'rgba(255, 99, 132, 1)', background: 'rgba(255, 99, 132, 0.2)' },
  { border: 'rgba(54, 162, 235, 1)', background: 'rgba(54, 162, 235, 0.2)' }
];

// Función para obtener datos desde un endpoint PHP con manejo de errores
async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) {
      throw new Error(`Error en la petición HTTP: ${response.status}`);
    }
    return await response.json();
  } catch (error) {
    console.error(`Fetch error en ${url}:`, error);
    return null;
  }
}

// Inicialización de la gráfica de evolución (línea) con múltiples datasets
async function initEvolutionChart() {
  if (!userId) {
    console.error("ID de usuario no especificado.");
    return;
  }
  const evolutionData = await fetchData('get_evolutionIndHabAcaBas_data.php?id=' + userId);
  if (!evolutionData || evolutionData.length === 0) {
    console.error("No se pudieron obtener datos para la gráfica de evolución.");
    return;
  }
  
  const labels = evolutionData.map(item => item.mes);
  
  // Usamos la clave correcta "lectura_comprensión" (con acento)
  const aciertosMemoriaVerbal = evolutionData.map(item => item['lectura_comprensión'] ? item['lectura_comprensión'].total_score : 0);
  const aciertosEscritura = evolutionData.map(item => item.escritura ? item.escritura.total_score : 0);
  const aciertosMatematicas = evolutionData.map(item => item.matematicas_basicas ? item.matematicas_basicas.total_score : 0);

  const canvasLine = document.getElementById('evolutionChart');
  if (!canvasLine) {
    console.error("Elemento canvas 'evolutionChart' no encontrado.");
    return;
  }
  
  const ctxLine = canvasLine.getContext('2d');
  new Chart(ctxLine, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Comprensión Lectora',
          data: aciertosMemoriaVerbal,
          borderColor: colorPalette[0].border,
          backgroundColor: colorPalette[0].background,
          fill: true
        },
        {
          label: 'Escritura',
          data: aciertosEscritura,
          borderColor: colorPalette[1].border,
          backgroundColor: colorPalette[1].background,
          fill: true
        },
        {
          label: 'Matemáticas Básicas',
          data: aciertosMatematicas,
          borderColor: colorPalette[2].border,
          backgroundColor: colorPalette[2].background,
          fill: true
        }
      ]
    },
    options: {
      responsive: true,
      plugins: {
        zoom: {
          zoom: {
            wheel: { enabled: true },
            pinch: { enabled: true },
            mode: 'x'
          },
          pan: { enabled: true, mode: 'x' }
        }
      },
      scales: {
        x: { title: { display: true, text: 'Mes/Año' } },
        y: { title: { display: true, text: 'Puntaje' } }
      }
    }
  });
}


// Inicialización de la gráfica de torta (pie chart) con porcentajes en tooltips y en labels
async function initPieChart() {
  const pieData = await fetchData('get_pieIndHabAcaBas_data.php?id=' + userId);
  if (!pieData) {
    console.error("No se pudieron obtener datos para la gráfica de torta.");
    return;
  }
  
  const pieLabels = pieData.map(item => item.test);
  const pieValues = pieData.map(item => item.total_score);
  const totalscore = pieValues.reduce((acc, val) => acc + val, 0);
  
  const canvasPie = document.getElementById('pieChart');
  if (!canvasPie) {
    console.error('No se encontró el elemento canvas con id "pieChart".');
    return;
  }
  
  const ctxPie = canvasPie.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: [
          colorPalette[0].border,
          colorPalette[1].border,
          colorPalette[2].border
        ]
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const percentage = totalscore > 0 ? ((value / totalscore) * 100).toFixed(2) : 0;
              return `${label}: ${percentage}% (${value})`;
            }
          }
        },
        datalabels: {
          color: '#fff',
          formatter: function(value) {
            const percentage = totalscore > 0 ? ((value / totalscore) * 100).toFixed(2) : 0;
            return percentage + '%';
          }
        }
      }
    },
    plugins: [ChartDataLabels]
  });
}

window.onload = function() {
  initEvolutionChart();
  initPieChart();
};
