// archivo appIndIndCogTarAca.js
let evolutionChartInstance;
let pieChartInstance;
// Asegúrate de que en tu HTML exista un elemento oculto con id "participant_id"
const participantElem = document.getElementById('participant_id');
const userId = participantElem ? participantElem.value.trim() : '';
console.log("ID de usuario:", userId);

// Construir las URLs con el parámetro "id"
const evolutionEndpoint = userId ? 'get_evolutionIndIndCogTarAca_data.php?id=' + userId : 'get_evolutionIndIndCogTarAca_data.php';
const pieEndpoint = userId ? 'get_pieIndIndCogTarAca_data.php?id=' + userId : 'get_pieIndIndCogTarAca_data.php';

const colorPalette = [
  { border: 'rgba(75, 192, 192, 1)', background: 'rgba(75, 192, 192, 0.2)' },
  { border: 'rgba(255, 99, 132, 1)', background: 'rgba(255, 99, 132, 0.2)' },
  { border: 'rgba(255, 206, 86, 1)', background: 'rgba(255, 206, 86, 0.2)' }
];

async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) {
      throw new Error(`Error en la petición HTTP: ${response.status}`);
    }
    const json = await response.json();
    console.log("Datos recibidos de", url, ":", json);
    return json;
  } catch (error) {
    console.error(`Fetch error en ${url}:`, error);
    return null;
  }
}

async function initEvolutionChart() {
  const evolutionData = await fetchData(evolutionEndpoint);
  console.log("Datos de evolución (IndIndCogTarAca):", evolutionData);
  if (!evolutionData || evolutionData.length === 0) {
    console.error("No se pudieron obtener datos para la gráfica de evolución.");
    return;
  }
  
  // Extraer etiquetas (meses)
  const labels = evolutionData.map(item => item.mes);
  
  // Extraer datos usando las claves reales que devuelve el JSON
  // Se asume que el endpoint devuelve claves: "atencion_concentracion", "resolucion_problemas" y "resultados_retencion"
  const scoreAtencion = evolutionData.map(item => item.atencion_concentracion ? item.atencion_concentracion.total_aciertos : 0);
  const scoreResolucion = evolutionData.map(item => item.resolucion_problemas ? item.resolucion_problemas.total_aciertos : 0);
  const scoreRetencion = evolutionData.map(item => item.resultados_retencion ? item.resultados_retencion.total_aciertos : 0);

  const canvasLine = document.getElementById('evolutionChart');
  if (!canvasLine) {
    console.error("No se encontró el elemento canvas con id 'evolutionChart'.");
    return;
  }
  
  const ctxLine = canvasLine.getContext('2d');
  new Chart(ctxLine, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Atención y Concentración',
          data: scoreAtencion,
          borderColor: colorPalette[0].border,
          backgroundColor: colorPalette[0].background,
          fill: true,
        },
        {
          label: 'Resolución de Problemas',
          data: scoreResolucion,
          borderColor: colorPalette[1].border,
          backgroundColor: colorPalette[1].background,
          fill: true,
        },
        {
          label: 'Retención de Información',
          data: scoreRetencion,
          borderColor: colorPalette[2].border,
          backgroundColor: colorPalette[2].background,
          fill: true,
        }
      ]
    },
    options: {
      responsive: true,
      plugins: {
        zoom: {
          zoom: {
            wheel: { enabled: true },
            pinch: { enabled: true },
            mode: 'x'
          },
          pan: { enabled: true, mode: 'x' }
        }
      },
      scales: {
        x: { title: { display: true, text: 'Mes/Año' } },
        y: { title: { display: true, text: 'Puntaje' } }
      }
    }
  });
}

async function initPieChart() {
  const pieData = await fetchData(pieEndpoint);
  console.log("Datos pie (IndIndCogTarAca):", pieData);
  if (!pieData || pieData.length === 0) {
    console.error("No se pudieron obtener datos para la gráfica de torta.");
    return;
  }
  
  const pieLabels = pieData.map(item => item.test);
  const pieValues = pieData.map(item => item.total_aciertos);
  const total = pieValues.reduce((acc, val) => acc + val, 0);
  
  const canvasPie = document.getElementById('pieChart');
  if (!canvasPie) {
    console.error("No se encontró el elemento canvas con id 'pieChart'.");
    return;
  }
  
  const ctxPie = canvasPie.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: [
          'rgba(75, 192, 192, 0.6)',
          'rgba(255, 99, 132, 0.6)',
          'rgba(255, 206, 86, 0.6)'
        ]
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const percentage = total > 0 ? ((value / total) * 100).toFixed(2) : 0;
              return `${label}: ${percentage}% (${value})`;
            }
          }
        }
      }
    }
  });
}

window.onload = function() {
  initEvolutionChart();
  initPieChart();
};
