<?php
// archivo get_evolutionIndCogTarAca_data.php

header('Content-Type: application/json');
include 'db_config.php';

// Arreglo para almacenar la información agrupada por mes.
$data = [];

/**
 * Procesa los datos de una tabla agrupados por mes.
 *
 * Realiza una consulta que agrupa los registros por mes (formateado como 'YYYY-MM') y obtiene la suma de aciertos y el total de registros.
 *
 * @param mysqli $conn Conexión a la base de datos.
 * @param string $table Nombre de la tabla.
 * @param string $dateColumn Nombre de la columna que contiene la fecha.
 * @param string $aciertosColumn Nombre de la columna que contiene los aciertos.
 * @param string $testName Identificador del test para indexar los resultados.
 * @return void
 */
function processTable(mysqli $conn, string $table, string $dateColumn, string $aciertosColumn, string $testName): void {
    global $data;

    // Preparar la consulta SQL que agrupa por mes.
    $sql = "SELECT DATE_FORMAT($dateColumn, '%Y-%m') AS mes, 
                   SUM($aciertosColumn) AS total_aciertos, 
                   COUNT(*) AS total_tests 
            FROM $table 
            GROUP BY mes 
            ORDER BY mes ASC";

    // Ejecutar la consulta y validar su ejecución.
    if ($result = $conn->query($sql)) {
        while ($row = $result->fetch_assoc()) {
            $mes = $row['mes'];
            if (!isset($data[$mes])) {
                $data[$mes] = [];
            }
            $data[$mes][$testName] = [
                'total_aciertos' => (int)$row['total_aciertos'],
                'total_tests'    => (int)$row['total_tests']
            ];
        }
        $result->free();
    } else {
        // Registrar error con información adicional
        error_log("Error en la consulta para la tabla '$table' con SQL [$sql]: " . $conn->error);
    }
}

// Procesar cada tabla
processTable($conn, 'atencion_concentracion', 'fecha_registro', 'atencion_concentracion', 'atencion_concentracion');
processTable($conn, 'resolucion_problemas', 'fecha_registro', 'aciertos', 'resolucion_problemas');
// Para las tablas resultados_retencion y visuoespacial_results se asume que el campo de fecha es 'fecha_registro'
processTable($conn, 'resultados_retencion', 'fecha_registro', 'aciertos', 'resultados_retencion');

// Verificar si $data quedó vacío (sin registros)
if (empty($data)) {
    error_log("No se obtuvieron datos. Verifica que las tablas contengan registros y que las columnas de fecha y aciertos tengan valores válidos.");
}

// Transformar el arreglo asociativo en un arreglo de objetos ordenados por mes.
ksort($data);
$finalData = [];
foreach ($data as $mes => $tests) {
    $entry = ['mes' => $mes];
    $entry['atencion_concentracion'] = isset($tests['atencion_concentracion']) ? $tests['atencion_concentracion'] : ['total_aciertos' => 0, 'total_tests' => 0];
    $entry['resolucion_problemas'] = isset($tests['resolucion_problemas']) ? $tests['resolucion_problemas'] : ['total_aciertos' => 0, 'total_tests' => 0];
    $entry['resultados_retencion'] = isset($tests['resultados_retencion']) ? $tests['resultados_retencion'] : ['total_aciertos' => 0, 'total_tests' => 0];
    $finalData[] = $entry;
}

echo json_encode($finalData);
?>
