<?php
// archivo get_evolution_data.php
header('Content-Type: application/json');
include 'db_config.php';

// Arreglo para almacenar la información agrupada por mes.
$data = [];

/**
 * Procesa los datos de una tabla agrupados por mes.
 *
 * Realiza una consulta que agrupa los registros por mes (formateado como 'YYYY-MM') y obtiene la suma de aciertos y el total de registros.
 *
 * @param mysqli $conn Conexión a la base de datos.
 * @param string $table Nombre de la tabla.
 * @param string $dateColumn Nombre de la columna que contiene la fecha.
 * @param string $aciertosColumn Nombre de la columna que contiene los aciertos.
 * @param string $testName Identificador del test para indexar los resultados.
 * @return void
 */
function processTable(mysqli $conn, string $table, string $dateColumn, string $aciertosColumn, string $testName): void {
    global $data;

    // Preparar la consulta SQL que agrupa por mes.
    $sql = "SELECT DATE_FORMAT($dateColumn, '%Y-%m') AS mes, 
                   SUM($aciertosColumn) AS total_aciertos, 
                   COUNT(*) AS total_tests 
            FROM $table 
            GROUP BY mes 
            ORDER BY mes ASC";

    // Ejecutar la consulta y validar su ejecución.
    if ($result = $conn->query($sql)) {
        while ($row = $result->fetch_assoc()) {
            $mes = $row['mes'];
            if (!isset($data[$mes])) {
                $data[$mes] = [];
            }
            $data[$mes][$testName] = [
                'total_aciertos' => (int)$row['total_aciertos'],
                'total_tests'    => (int)$row['total_tests']
            ];
        }
        $result->free();
    } else {
        // Registrar error con información adicional
        error_log("Error en la consulta para la tabla '$table' con SQL [$sql]: " . $conn->error);
    }
}

// Procesar cada tabla
processTable($conn, 'test_entries', 'created_at', 'attempts', 'test_entries');
processTable($conn, 'funciones_ejecutivas', 'fecha_registro', 'clasificaciones_correctas', 'funciones_ejecutivas');
// Para las tablas atencion_velocidad_procesamiento y visuoespacial_results se asume que el campo de fecha es 'fecha_registro'
processTable($conn, 'atencion_velocidad_procesamiento', 'fecha_registro', 'aciertos', 'atencion_velocidad_procesamiento');
processTable($conn, 'visuoespacial_results', 'fecha_registro', 'aciertos', 'visuoespacial_results');

// Verificar si $data quedó vacío (sin registros)
if (empty($data)) {
    error_log("No se obtuvieron datos. Verifica que las tablas contengan registros y que las columnas de fecha y aciertos tengan valores válidos.");
}

// Transformar el arreglo asociativo en un arreglo de objetos ordenados por mes.
ksort($data);
$finalData = [];
foreach ($data as $mes => $tests) {
    $entry = ['mes' => $mes];
    $entry['test_entries'] = isset($tests['test_entries']) ? $tests['test_entries'] : ['total_aciertos' => 0, 'total_tests' => 0];
    $entry['funciones_ejecutivas'] = isset($tests['funciones_ejecutivas']) ? $tests['funciones_ejecutivas'] : ['total_aciertos' => 0, 'total_tests' => 0];
    $entry['atencion_velocidad_procesamiento'] = isset($tests['atencion_velocidad_procesamiento']) ? $tests['atencion_velocidad_procesamiento'] : ['total_aciertos' => 0, 'total_tests' => 0];
    $entry['visuoespacial_results'] = isset($tests['visuoespacial_results']) ? $tests['visuoespacial_results'] : ['total_aciertos' => 0, 'total_tests' => 0];
    $finalData[] = $entry;
}

echo json_encode($finalData);
?>
