// mobileAppAnaEstDesAteCon.js

async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) throw new Error(`HTTP error ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error("Error fetching data:", error);
    return null;
  }
}

async function initCharts() {
  const data = await fetchData('get_lineAnaEstDesAteConChartData.php');
  if (!data) {
    console.error("No se recibieron datos desde el PHP.");
    return;
  }
  console.log("Datos recibidos:", data);
  
  const lineData = data.lineData;  // Objeto: modalidad => { mes: total, ... }
  const pieData  = data.pieData;    // Objeto: modalidad => total
  
  // Modalidades a graficar (sin acentos)
  const modalities = [
    "baja intensidad y equilibrio",
    "resistencia moderada",
    "aerobicos adaptados",
    "ninguna"
  ];
  
  // Recolectar todos los meses (labels) presentes en cualquiera de las modalidades
  const monthsSet = new Set();
  modalities.forEach(mod => {
    if (lineData[mod]) {
      Object.keys(lineData[mod]).forEach(m => monthsSet.add(m));
    }
  });
  let months = Array.from(monthsSet).sort();
  if (months.length === 0) {
    months = ["Sin datos"];
  }
  
  // Colores para cada modalidad
  const colors = {
    "baja intensidad y equilibrio": 'rgba(255, 99, 132, 1)',
    "resistencia moderada": 'rgba(54, 162, 235, 1)',
    "aerobicos adaptados": 'rgba(255, 206, 86, 1)', // Amarillo
    "ninguna": 'rgba(75, 192, 192, 1)'
  };
  
  // Preparar datasets para la gráfica de líneas: cada modalidad tendrá una línea
  const datasets = modalities.map(mod => {
    const dataPoints = months.map(month => {
      return (lineData[mod] && lineData[mod][month] !== undefined) ? lineData[mod][month] : 0;
    });
    return {
      label: mod.toUpperCase(),
      data: dataPoints,
      borderColor: colors[mod],
      backgroundColor: colors[mod],
      fill: false,
      tension: 0.1
    };
  });
  
  // Seleccionar el contenedor donde se insertarán las gráficas
  const chartsContainer = document.getElementById('chartsContainer');
  if (!chartsContainer) {
    console.error("No se encontró el contenedor con id 'chartsContainer'.");
    return;
  }
  chartsContainer.innerHTML = ""; // Limpiar contenedor
  
  // --- Gráfico de líneas ---
  const lineDiv = document.createElement('div');
  lineDiv.className = 'chart-container';
  
  const lineTitle = document.createElement('h3');
  lineTitle.textContent = "Evolución Mensual - Sumatoria de Aciertos";
  lineDiv.appendChild(lineTitle);
  
  const lineCanvas = document.createElement('canvas');
  lineCanvas.id = "lineChart";
  lineCanvas.style.height = "400px";
  lineCanvas.style.width = "100%";
  lineDiv.appendChild(lineCanvas);
  chartsContainer.appendChild(lineDiv);
  
  const ctxLine = lineCanvas.getContext('2d');
  new Chart(ctxLine, {
    type: 'line',
    data: {
      labels: months,
      datasets: datasets
    },
    options: {
      responsive: true,
      plugins: {
        title: {
          display: true,
          text: "Evolución Mensual - Sumatoria de Aciertos"
        }
      },
      scales: {
        x: { title: { display: true, text: 'Mes (YYYY-MM)' } },
        y: { title: { display: true, text: 'Aciertos Totales' } }
      }
    }
  });
  
  // --- Gráfico de torta ---
  const pieDiv = document.createElement('div');
  pieDiv.className = 'chart-container';
  
  const pieTitle = document.createElement('h3');
  pieTitle.textContent = "Acumulado Total por Modalidad";
  pieDiv.appendChild(pieTitle);
  
  const pieCanvas = document.createElement('canvas');
  pieCanvas.id = "pieChart";
  pieCanvas.style.height = "400px";
  pieCanvas.style.width = "100%";
  pieDiv.appendChild(pieCanvas);
  chartsContainer.appendChild(pieDiv);
  
  // Preparar datos para la gráfica de torta
  const pieLabels = modalities.map(mod => mod.toUpperCase());
  const pieValues = modalities.map(mod => pieData[mod] || 0);
  const pieColors = [
    'rgba(255, 99, 132, 0.6)',    // baja intensidad y equilibrio
    'rgba(54, 162, 235, 0.6)',     // resistencia moderada
    'rgba(255, 206, 86, 0.6)',     // aerobicos adaptados (amarillo)
    'rgba(75, 192, 192, 0.6)'      // ninguna
  ];
  
  const ctxPie = pieCanvas.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: pieColors
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        title: { display: true, text: "Acumulado Total por Modalidad" },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const total = pieValues.reduce((sum, val) => sum + val, 0);
              const percentage = total > 0 ? ((value / total) * 100).toFixed(2) : 0;
              return `${label}: ${percentage}% (${value})`;
            }
          }
        }
      }
    }
  });
}

window.onload = initCharts;
