
//mobileAppAnaEstDesGloHabAcaBas.js

async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) throw new Error(`HTTP error ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error("Error fetching data:", error);
    return null;
  }
} 

async function initCharts() {
  const rawData = await fetchData('get_lineEducacionChartData.php');
  console.log("Datos recibidos:", rawData);
  if (!rawData) {
    console.error("No se pudieron obtener datos para las gráficas.");
    return;
  }
  
  const lineData = rawData.lineData;
  const pieData  = rawData.pieData;
  
  // Modalidades (las tres específicas y "ninguna")
  const modalities = [
    "baja intensidad y equilibrio",
    "resistencia moderada",
    "aerobicos adaptados",
    "ninguna"
  ];
  
  // Tests a graficar
  const tests = [
    "Comprensión Lectora",
    "Escritura",
    "Matemáticas Básicas"
  ];
  
  // Contenedor donde se crearán los gráficos
  const chartsContainer = document.getElementById('chartsContainer');
  if (!chartsContainer) {
    console.error("No se encontró el contenedor con id 'chartsContainer'.");
    return;
  }
  chartsContainer.innerHTML = "";
  
  // Colores para cada test
  const testColors = [
    'rgba(255, 99, 132, 1)',    // Comprensión Lectora
    'rgba(54, 162, 235, 1)',    // Escritura
    'rgba(255, 206, 86, 1)'     // Matemáticas Básicas
  ];
  
  // --- Crear gráfico de línea para cada modalidad ---
  modalities.forEach(mod => {
    // Contenedor para el gráfico
    const lineDiv = document.createElement('div');
    lineDiv.className = 'chart-container';
    lineDiv.style.width = "800px";
    lineDiv.style.margin = "20px auto";
    
    // Título
    const title = document.createElement('h3');
    title.textContent = mod.toUpperCase() + " - Puntaje";
    lineDiv.appendChild(title);
    
    // Canvas para el gráfico de línea
    const canvas = document.createElement('canvas');
    canvas.id = mod.replace(/\s/g, '') + 'LineChart';
    canvas.style.height = "400px";
    canvas.style.width = "100%";
    lineDiv.appendChild(canvas);
    chartsContainer.appendChild(lineDiv);
    
    // Reunir etiquetas (meses) para la modalidad (unión de los meses de cada test)
    const monthsSet = new Set();
    tests.forEach(test => {
      if (lineData[test] && lineData[test][mod]) {
        Object.keys(lineData[test][mod]).forEach(month => monthsSet.add(month));
      }
    });
    let labels = Array.from(monthsSet).sort();
    if (labels.length === 0) {
      labels = ["Sin datos"];
    }
    
    // Preparar datasets para cada test
    const datasets = tests.map((test, index) => {
      const testData = (lineData[test] && lineData[test][mod]) ? lineData[test][mod] : {};
      const dataPoints = labels.map(month => (testData[month] !== undefined ? testData[month] : null));
      return {
        label: test,
        data: dataPoints,
        borderColor: testColors[index],
        fill: false,
        tension: 0.1
      };
    });
    
    // Dibujar gráfico de línea
    const ctx = canvas.getContext('2d');
    new Chart(ctx, {
      type: 'line',
      data: {
        labels: labels,
        datasets: datasets
      },
      options: {
        responsive: true,
        plugins: {
          title: {
            display: true,
            text: mod.toUpperCase() + " -  Puntaje"
          }
        },
        scales: {
          x: { title: { display: true, text: 'Mes (YYYY-MM)' } },
          y: { title: { display: true, text: 'Puntaje' } }
        }
      }
    });
  });
  
  // --- Crear gráfico de torta para cada modalidad ---
  modalities.forEach(mod => {
    // Contenedor para el gráfico de torta
    const pieDiv = document.createElement('div');
    pieDiv.className = 'chart-container';
    pieDiv.style.width = "800px";
    pieDiv.style.margin = "20px auto";
    
    const pieTitle = document.createElement('h3');
    pieTitle.textContent = mod.toUpperCase() + " - Acumulados Totales por Test";
    pieDiv.appendChild(pieTitle);
    
    const pieCanvas = document.createElement('canvas');
    pieCanvas.id = mod.replace(/\s/g, '') + 'PieChart';
    pieCanvas.style.height = "400px";
    pieCanvas.style.width = "100%";
    pieDiv.appendChild(pieCanvas);
    chartsContainer.appendChild(pieDiv);
    
    // Para cada test, se toma el total acumulado para la modalidad
    const totals = tests.map(test => {
      let total = 0;
      if (pieData[mod] && pieData[mod][test]) {
        total = pieData[mod][test];
      }
      return total;
    });
    
    const ctxPie = pieCanvas.getContext('2d');
    new Chart(ctxPie, {
      type: 'pie',
      data: {
        labels: tests,
        datasets: [{
          data: totals,
          backgroundColor: testColors.map(color => color.replace(', 1)', ', 0.6)'))
        }]
      },
      options: {
        responsive: true,
        plugins: {
          legend: { position: 'top' },
          title: { display: true, text: mod.toUpperCase() + " - Acumulados Totales por Test" },
          tooltip: {
            callbacks: {
              label: function(context) {
                const label = context.label || '';
                const value = context.raw;
                const sum = totals.reduce((a, b) => a + b, 0);
                const pct = sum ? ((value / sum) * 100).toFixed(2) : 0;
                return `${label}: ${pct}% (${value})`;
              }
            }
          }
        }
      }
    });
  });
}

window.onload = initCharts;
