// mobileAppAnaEstDesGloHabArt.js

async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) throw new Error(`HTTP error ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error("Error fetching", url, error);
    return null;
  }
}

async function initCharts() {
  // Obtener los datos de los tests de artes
  const rawData = await fetchData('get_lineArtesChartData.php');
  console.log("Datos recibidos:", rawData);
  if (!rawData) {
    console.error("No se pudieron obtener datos para las gráficas.");
    return;
  }
  
  // Definir las 4 modalidades (las tres específicas y "ninguna")
  const modalities = [
    "baja intensidad y equilibrio",
    "resistencia moderada",
    "aerobicos adaptados",
    "ninguna"
  ];
  
  // Definir los nombres de los 5 tests (según los utilizados en el PHP)
  const tests = [
    "Dibujo y Pintura",
    "Escultura y Modelado",
    "Ritmo y Tempo",
    "Danza y Expresión Corporal",
    "Expresión Teatral"
  ];
  
  // Seleccionar el contenedor donde se ubicarán los gráficos
  const chartsContainer = document.getElementById('chartsContainer');
  if (!chartsContainer) {
    console.error("No se encontró el contenedor de gráficos con id 'chartsContainer'.");
    return;
  }
  chartsContainer.innerHTML = "";
  
  // Colores para cada test (5 colores)
  const testColors = [
    'rgba(255, 99, 132, 1)',    // Dibujo y Pintura
    'rgba(54, 162, 235, 1)',    // Escultura y Modelado
    'rgba(255, 206, 86, 1)',    // Ritmo y Tempo
    'rgba(75, 192, 192, 1)',    // Danza y Expresión Corporal
    'rgba(153, 102, 255, 1)'    // Expresión Teatral
  ];
  
  // --- Crear gráficos de línea (una por modalidad) ---
  modalities.forEach(mod => {
    // Crear un contenedor para el gráfico de líneas
    const lineDiv = document.createElement('div');
    lineDiv.className = 'chart-container';
    
    const title = document.createElement('h3');
    title.textContent = mod.toUpperCase() + " - Línea de Tiempo";
    lineDiv.appendChild(title);
    
    const canvas = document.createElement('canvas');
    canvas.id = mod.replace(/\s/g, '') + 'LineChart';
    lineDiv.appendChild(canvas);
    chartsContainer.appendChild(lineDiv);
    
    // Para esta modalidad, reunir todos los meses (labels) de los 5 tests
    const monthsSet = new Set();
    tests.forEach(test => {
      if (rawData[test] && rawData[test][mod]) {
        Object.keys(rawData[test][mod]).forEach(m => monthsSet.add(m));
      }
    });
    let labels = Array.from(monthsSet).sort();
    if (labels.length === 0) {
      labels = ["Sin datos"];
    }
    
    // Crear datasets: uno por cada test
    const datasets = tests.map((test, i) => {
      const testData = (rawData[test] && rawData[test][mod]) ? rawData[test][mod] : {};
      const dataPoints = labels.map(month => testData[month] || 0);
      return {
        label: test,
        data: dataPoints,
        borderColor: testColors[i],
        fill: false,
        tension: 0.1
      };
    });
    
    const ctx = canvas.getContext('2d');
    new Chart(ctx, {
      type: 'line',
      data: {
        labels: labels,
        datasets: datasets
      },
      options: {
        responsive: true,
        plugins: {
          title: { display: true, text: mod.toUpperCase() + " - Evolución Mensual" }
        },
        scales: {
          x: { title: { display: true, text: 'Mes (YYYY-MM)' } },
          y: { title: { display: true, text: 'Promedio' } }
        }
      }
    });
  });
  
  // --- Crear gráficos de torta (una por modalidad) ---
  // Para cada modalidad, se calcula el acumulado (suma de todos los valores mensuales) para cada test
  modalities.forEach(mod => {
    // Crear un contenedor para la gráfica de torta
    const pieDiv = document.createElement('div');
    pieDiv.className = 'chart-container';
    
    const pieTitle = document.createElement('h3');
    pieTitle.textContent = mod.toUpperCase() + " - Acumulados Totales";
    pieDiv.appendChild(pieTitle);
    
    const pieCanvas = document.createElement('canvas');
    pieCanvas.id = mod.replace(/\s/g, '') + 'PieChart';
    pieDiv.appendChild(pieCanvas);
    chartsContainer.appendChild(pieDiv);
    
    // Para cada test, sumar los valores de todos los meses para esta modalidad
    const totals = tests.map(test => {
      let total = 0;
      if (rawData[test] && rawData[test][mod]) {
        Object.values(rawData[test][mod]).forEach(val => {
          total += val;
        });
      }
      return total;
    });
    
    const ctxPie = pieCanvas.getContext('2d');
    new Chart(ctxPie, {
      type: 'pie',
      data: {
        labels: tests,
        datasets: [{
          data: totals,
          backgroundColor: testColors.map(color => color.replace(', 1)', ', 0.6)'))
        }]
      },
      options: {
        responsive: true,
        plugins: {
          legend: { position: 'top' },
          title: { display: true, text: mod.toUpperCase() + " - Acumulados por Test" },
          tooltip: {
            callbacks: {
              label: function(context) {
                const label = context.label || '';
                const value = context.raw;
                const sum = totals.reduce((a, b) => a + b, 0);
                const pct = sum ? ((value / sum) * 100).toFixed(2) : 0;
                return `${label}: ${pct}% (${value})`;
              }
            }
          }
        }
      }
    });
  });
}

window.onload = initCharts;
