// mobileAppAnaEstDesGloHabDibPin.js

async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) throw new Error(`HTTP error ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error("Error fetching", url, error);
    return null;
  }
}

async function initCharts() {
  // Obtener los datos desde el PHP
  const rawData = await fetchData('get_lineDibujoPinturaChartData.php');
  console.log("Datos recibidos:", rawData);
  if (!rawData) {
    console.error("No se pudieron obtener datos para las gráficas.");
    return;
  }
  
  // Definir las 4 modalidades (las tres específicas y "ninguna")
  const modalities = [
    "baja intensidad y equilibrio",
    "resistencia moderada",
    "aerobicos adaptados",
    "ninguna"
  ];
  
  // Definir los nombres de las 3 métricas (líneas)
  const metrics = [
    "Puntaje Precision",
    "Puntaje Colores",
    "Puntaje Organizacion"
  ];
  
  // Seleccionar el contenedor donde se insertarán los gráficos
  const chartsContainer = document.getElementById('chartsContainer');
  if (!chartsContainer) {
    console.error("No se encontró el contenedor de gráficos con id 'chartsContainer'.");
    return;
  }
  chartsContainer.innerHTML = ""; // Limpiar contenedor
  
  // Colores para cada métrica
  const metricColors = [
    'rgba(255, 99, 132, 1)',    // Rojo
    'rgba(54, 162, 235, 1)',    // Azul
    'rgba(255, 206, 86, 1)'     // Amarillo
  ];
  
  // --- Crear gráficos de línea para cada modalidad ---
  modalities.forEach(mod => {
    // Crear contenedor para el gráfico de línea
    const modDiv = document.createElement('div');
    modDiv.className = 'chart-container';
    
    const title = document.createElement('h3');
    title.textContent = mod.toUpperCase() + " - Evolución Mensual";
    modDiv.appendChild(title);
    
    const canvas = document.createElement('canvas');
    canvas.id = mod.replace(/\s/g, '') + 'LineChart';
    modDiv.appendChild(canvas);
    chartsContainer.appendChild(modDiv);
    
    // Para la modalidad actual, obtener la unión de todos los meses presentes en los datos de las 3 métricas
    const monthsSet = new Set();
    metrics.forEach(metric => {
      if (rawData[metric] && rawData[metric][mod]) {
        Object.keys(rawData[metric][mod]).forEach(m => monthsSet.add(m));
      }
    });
    let labels = Array.from(monthsSet).sort();
    if (labels.length === 0) {
      labels = ["Sin datos"];
    }
    
    // Crear datasets: uno por cada métrica
    const datasets = metrics.map((metric, i) => {
      const metricData = (rawData[metric] && rawData[metric][mod]) ? rawData[metric][mod] : {};
      const dataPoints = labels.map(month => metricData[month] || 0);
      return {
        label: metric,
        data: dataPoints,
        borderColor: metricColors[i],
        fill: false,
        tension: 0.1
      };
    });
    
    const ctx = canvas.getContext('2d');
    new Chart(ctx, {
      type: 'line',
      data: {
        labels: labels,
        datasets: datasets
      },
      options: {
        responsive: true,
        plugins: {
          title: { display: true, text: mod.toUpperCase() + " - Evolución Mensual" }
        },
        scales: {
          x: { title: { display: true, text: 'Mes (YYYY-MM)' } },
          y: { title: { display: true, text: 'Acumulado de Puntajes' } }
        }
      }
    });
  });
  
  // --- Crear gráfica de torta (pie chart) de acumulados generales ---
  // Se calculará, para cada modalidad, la suma total de los tres puntajes (acumulado a lo largo del período)
  const pieTotals = {
    "baja intensidad y equilibrio": 0,
    "resistencia moderada": 0,
    "aerobicos adaptados": 0,
    "ninguna": 0
  };
  
  // Recorremos rawData para acumular totales por modalidad
  metrics.forEach(metric => {
    if (rawData[metric]) {
      Object.keys(rawData[metric]).forEach(mod => {
        Object.values(rawData[metric][mod]).forEach(val => {
          pieTotals[mod] += val;
        });
      });
    }
  });
  
  // Crear contenedor para la gráfica de torta
  const pieDiv = document.createElement('div');
  pieDiv.className = 'chart-container';
  
  const pieTitle = document.createElement('h3');
  pieTitle.textContent = "Acumulados Generales por Modalidad";
  pieDiv.appendChild(pieTitle);
  
  const pieCanvas = document.createElement('canvas');
  pieCanvas.id = 'globalPieChart';
  pieDiv.appendChild(pieCanvas);
  chartsContainer.appendChild(pieDiv);
  
  const pieLabels = Object.keys(pieTotals);
  const pieValues = pieLabels.map(mod => pieTotals[mod]);
  const pieColors = [
    'rgba(255, 99, 132, 0.6)',    // baja intensidad y equilibrio
    'rgba(54, 162, 235, 0.6)',     // resistencia moderada
    'rgba(255, 206, 86, 0.6)',     // aerobicos adaptados
    'rgba(75, 192, 192, 0.6)'      // ninguna
  ];
  
  const ctxPie = pieCanvas.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: pieColors
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        title: { display: true, text: 'Acumulados Generales por Modalidad' },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const total = pieValues.reduce((acc, val) => acc + val, 0);
              const pct = total > 0 ? ((value / total) * 100).toFixed(2) : 0;
              return `${label}: ${pct}% (${value})`;
            }
          }
        }
      }
    }
  });
}

window.onload = initCharts;
