// mobileAppAnaEstDesGloObsReg.js

async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) throw new Error(`HTTP error ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error("Error fetching", url, error);
    return null;
  }
}

async function initCharts() {
  // Obtener los datos agrupados por pregunta, modalidad y mes
  const rawData = await fetchData('get_lineAnaEstDesGloObsRegChartData.php');
  if (!rawData) {
    console.error("No se pudieron obtener datos para las gráficas.");
    return;
  }
  
  // Definir las 4 modalidades (las tres específicas y "ninguna")
  const modalities = [
    "baja intensidad y equilibrio",
    "resistencia moderada",
    "aerobicos adaptados",
    "ninguna"
  ];
  
  // Definir los nombres de las 10 preguntas (planos)
  const questionNames = [
    "Participación",
    "Atención",
    "Desafios",
    "Aprender",
    "Compañerismo",
    "Actitud",
    "Resolutividad",
    "Organización",
    "Resiliencia",
    "Responsabilidad"
  ];
  
  // Reorganizar los datos para estructurarlos por modalidad.
  // rawData tiene la estructura: 
  // { "Participación": { modalidad: { mes: valor, ... }, ... }, "Atención": { ... }, ... }
  // Creamos un objeto en el que cada clave sea una modalidad y su valor sea un objeto con cada pregunta.
  const modalityData = {};
  modalities.forEach(mod => {
    modalityData[mod] = {};
    questionNames.forEach(q => {
      modalityData[mod][q] = rawData[q] && rawData[q][mod] ? rawData[q][mod] : {};
    });
  });
  
  // Seleccionar el contenedor donde se ubicarán los gráficos
  const chartsContainer = document.getElementById('chartsContainer');
  if (!chartsContainer) {
    console.error("No se encontró el contenedor de gráficos con id 'chartsContainer'.");
    return;
  }
  chartsContainer.innerHTML = ""; // Limpiar el contenedor
  
  // Definir colores para las 10 líneas (preguntas)
  const questionColors = [
    'rgba(255, 99, 132, 1)',    // Participación
    'rgba(54, 162, 235, 1)',    // Atención
    'rgba(255, 206, 86, 1)',    // Desafios
    'rgba(75, 192, 192, 1)',    // Aprender
    'rgba(153, 102, 255, 1)',   // Compañerismo
    'rgba(255, 159, 64, 1)',    // Actitud
    'rgba(199, 199, 199, 1)',   // Resolutividad
    'rgba(83, 102, 255, 1)',    // Organización
    'rgba(255, 102, 255, 1)',   // Resiliencia
    'rgba(102, 255, 102, 1)'    // Responsabilidad
  ];
  
  // Para cada modalidad, generar un gráfico de línea
  modalities.forEach(mod => {
    // Crear un contenedor para el gráfico
    const modDiv = document.createElement('div');
    modDiv.className = 'chart-container';
    
    // Agregar un título con el nombre de la modalidad
    const modTitle = document.createElement('h3');
    modTitle.textContent = mod.toUpperCase();
    modDiv.appendChild(modTitle);
    
    // Crear el canvas para el gráfico
    const canvas = document.createElement('canvas');
    canvas.id = mod.replace(/\s/g, '') + 'Chart';
    modDiv.appendChild(canvas);
    chartsContainer.appendChild(modDiv);
    
    // Para la modalidad actual, obtener todos los meses que aparecen en cualquiera de las preguntas
    const monthsSet = new Set();
    questionNames.forEach(q => {
      const qData = modalityData[mod][q];
      Object.keys(qData).forEach(m => monthsSet.add(m));
    });
    const labels = Array.from(monthsSet).sort();
    
    // Crear los datasets: uno por cada pregunta
    const datasets = questionNames.map((q, index) => {
      const dataPoints = labels.map(month => {
        const qData = modalityData[mod][q];
        return qData && qData[month] ? qData[month] : 0;
      });
      return {
        label: q,
        data: dataPoints,
        borderColor: questionColors[index],
        fill: false,
        tension: 0.1
      };
    });
    
    // Crear la gráfica de línea para la modalidad actual
    const ctx = canvas.getContext('2d');
    new Chart(ctx, {
      type: 'line',
      data: {
        labels: labels,
        datasets: datasets
      },
      options: {
        responsive: true,
        plugins: {
          title: {
            display: true,
            text: mod.toUpperCase() + " - Evolución Mensual por Pregunta"
          }
        },
        scales: {
          x: { title: { display: true, text: 'Mes (YYYY-MM)' } },
          y: { title: { display: true, text: 'Acumulado de Puntajes' } }
        }
      }
    });
  });
  
  // --- Generar la gráfica de torta (pie chart) con los acumulativos generales de TODAS las preguntas ---
  // Se calculará, para cada modalidad, la suma total de puntajes de todas las preguntas y de todos los meses.
  const pieTotals = {
    "baja intensidad y equilibrio": 0,
    "resistencia moderada": 0,
    "aerobicos adaptados": 0,
    "ninguna": 0
  };
  
  // Recorrer rawData para acumular totales
  for (let question in rawData) {
    for (let mod in rawData[question]) {
      for (let month in rawData[question][mod]) {
        pieTotals[mod] += rawData[question][mod][month];
      }
    }
  }
  
  // Crear un contenedor para la gráfica de torta
  const pieDiv = document.createElement('div');
  pieDiv.className = 'chart-container';
  const pieTitle = document.createElement('h3');
  pieTitle.textContent = "Acumulativos Generales de Todas las Preguntas";
  pieDiv.appendChild(pieTitle);
  const pieCanvas = document.createElement('canvas');
  pieCanvas.id = 'globalPieChart';
  pieDiv.appendChild(pieCanvas);
  chartsContainer.appendChild(pieDiv);
  
  // Preparar datos para la gráfica de torta
  const pieLabels = Object.keys(pieTotals);
  const pieValues = pieLabels.map(mod => pieTotals[mod]);
  const pieColors = [
    'rgba(255, 99, 132, 0.6)',    // baja intensidad y equilibrio
    'rgba(54, 162, 235, 0.6)',     // resistencia moderada
    'rgba(255, 206, 86, 0.6)',     // aerobicos adaptados
    'rgba(75, 192, 192, 0.6)'      // ninguna
  ];
  
  const ctxPie = pieCanvas.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: pieColors
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        title: {
          display: true,
          text: 'Acumulativos Generales de Todas las Preguntas'
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const total = pieValues.reduce((acc, val) => acc + val, 0);
              const percentage = total > 0 ? ((value / total) * 100).toFixed(2) : 0;
              return `${label}: ${percentage}% (${value})`;
            }
          }
        }
      }
    }
  });
}

window.onload = initCharts;
