// mobileAppAnaEstDesGlobRenCog.js

async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) throw new Error(`HTTP error ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error("Error fetching data:", error);
    return null;
  }
}

async function initCharts() {
  console.log("Iniciando initCharts...");
  const data = await fetchData('get_lineAnaEstDesGlobRenCogChartData.php');
  if (!data) {
    console.error("No se recibieron datos desde el PHP.");
    return;
  }

  console.log("Datos recibidos:", data);

  const lineData = data.lineData;
  const pieData = data.pieData;

  // Modalidades a procesar: las tres específicas y "ninguna"
  const modalities = [
    "baja intensidad y equilibrio",
    "resistencia moderada",
    "aerobicos adaptados",
    "ninguna"
  ];
  // Los tests que se graficarán
  const tests = [
    "Memoria Verbal",
    "Funciones Ejecutivas",
    "Atención y Velocidad de Procesamiento",
    "Habilidades Visoespaciales"
  ];

  // Seleccionamos el contenedor donde se insertarán los gráficos
  const chartsContainer = document.getElementById('chartsContainer');
  if (!chartsContainer) {
    console.error("No se encontró el contenedor con id 'chartsContainer'.");
    return;
  }
  chartsContainer.innerHTML = "";

  // Definir colores para cada línea (test)
  const testColors = [
    'rgba(255, 99, 132, 1)',    // Memoria Verbal
    'rgba(54, 162, 235, 1)',    // Funciones Ejecutivas
    'rgba(255, 206, 86, 1)',    // Atención y Velocidad de Procesamiento
    'rgba(75, 192, 192, 1)'     // Habilidades Visoespaciales
  ];

  // --- Crear gráfico de línea para cada modalidad ---
  modalities.forEach(modality => {
    console.log("Creando gráfico de línea para modalidad:", modality);
    // Crear contenedor para el gráfico
    const containerDiv = document.createElement('div');
    containerDiv.className = 'chart-container';

    // Título para la modalidad
    const title = document.createElement('h3');
    title.textContent = modality.toUpperCase() + " - Puntaje";
    containerDiv.appendChild(title);

    // Crear canvas y asignar una altura para que se vea
    const canvas = document.createElement('canvas');
    canvas.id = modality.replace(/\s/g, '') + 'LineChart';
    canvas.style.height = "400px";
    containerDiv.appendChild(canvas);
    chartsContainer.appendChild(containerDiv);

    // Reunir todos los meses (labels) presentes en los datos de cualquiera de los tests
    const monthsSet = new Set();
    tests.forEach(test => {
      if (lineData[test] && lineData[test][modality]) {
        Object.keys(lineData[test][modality]).forEach(month => monthsSet.add(month));
      }
    });
    let labels = Array.from(monthsSet).sort();
    if (labels.length === 0) {
      labels = ["Sin datos"];
    }
    console.log("Etiquetas para", modality, ":", labels);

    // Preparar los datasets para cada test
    const datasets = tests.map((test, index) => {
      const testData = (lineData[test] && lineData[test][modality]) ? lineData[test][modality] : {};
      const dataPoints = labels.map(month => (testData[month] !== undefined ? testData[month] : null));
      return {
        label: test,
        data: dataPoints,
        borderColor: testColors[index],
        fill: false,
        tension: 0.1
      };
    });
    console.log("Datasets para", modality, ":", datasets);

    // Dibujar el gráfico de líneas
    const ctx = canvas.getContext('2d');
    new Chart(ctx, {
      type: 'line',
      data: {
        labels: labels,
        datasets: datasets
      },
      options: {
        responsive: true,
        plugins: {
          title: {
            display: true,
            text: modality.toUpperCase() + " - Puntaje"
          }
        },
        scales: {
          x: {
            title: { display: true, text: 'Mes (YYYY-MM)' }
          },
          y: {
            title: { display: true, text: 'Puntaje' }
          }
        }
      }
    });
  });

  // --- Crear gráfica de torta (pie chart) global ---
  console.log("Creando gráfico de torta global...");
  const pieTotals = {};
  modalities.forEach(modality => {
    let total = 0;
    tests.forEach(test => {
      if (pieData[modality] && pieData[modality][test]) {
        total += pieData[modality][test];
      }
    });
    pieTotals[modality] = total;
  });

  // Crear contenedor para la gráfica de torta
  const pieContainer = document.createElement('div');
  pieContainer.className = 'chart-container';
  const pieTitle = document.createElement('h3');
  pieTitle.textContent = "Acumulado por Modalidad";
  pieContainer.appendChild(pieTitle);

  const pieCanvas = document.createElement('canvas');
  pieCanvas.id = 'totalPieChart';
  pieCanvas.style.height = "400px";
  pieContainer.appendChild(pieCanvas);
  chartsContainer.appendChild(pieContainer);

  // Preparar datos para la gráfica de torta
  const pieLabels = modalities.map(mod => mod.toUpperCase());
  const pieValues = modalities.map(mod => pieTotals[mod]);
  const pieColors = [
    'rgba(255, 99, 132, 0.6)',
    'rgba(54, 162, 235, 0.6)',
    'rgba(255, 206, 86, 0.6)',
    'rgba(75, 192, 192, 0.6)'
  ];

  const ctxPie = pieCanvas.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: pieColors
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        title: { display: true, text: 'Acumulado por Modalidad' },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const total = pieValues.reduce((sum, val) => sum + val, 0);
              const percentage = total > 0 ? ((value / total) * 100).toFixed(2) : 0;
              return `${label}: ${percentage}% (${value})`;
            }
          }
        }
      }
    }
  });
}

window.onload = initCharts;
