// mobileAppAnaEstDesHabVis.js

async function fetchData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) throw new Error(`HTTP error ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error("Error fetching", url, error);
    return null;
  }
}

async function initCharts() {
  // Obtener datos de evolución (global)
  const evolutionData = await fetchData('get_lineAnaEstDesHabVisChartData.php');
  if (!evolutionData) {
    console.error("No se pudieron obtener datos para la gráfica de evolución.");
    return;
  }
  
  // Modalidades a graficar: las tres específicas y "ninguna"
  const modalities = [
    "ninguna",
    "baja intensidad y equilibrio",
    "resistencia moderada",
    "aerobicos adaptados"  // Sin acento para que coincida con el PHP
  ];
  
  // Recopilar todos los meses (labels) de todos los datasets
  let allMonths = new Set();
  modalities.forEach(mod => {
    if (evolutionData[mod]) {
      Object.keys(evolutionData[mod]).forEach(m => allMonths.add(m));
    }
  });
  const labels = Array.from(allMonths).sort();

  // Crear los datasets para cada modalidad
  const datasets = modalities.map((mod, index) => {
    let data = labels.map(m => evolutionData[mod] && evolutionData[mod][m] ? evolutionData[mod][m] : 0);
    let borderColor, backgroundColor;
    switch(index) {
      case 0:
        borderColor = 'rgba(75, 192, 192, 1)';
        backgroundColor = 'rgba(75, 192, 192, 0.2)';
        break;
      case 1:
        borderColor = 'rgba(255, 99, 132, 1)';
        backgroundColor = 'rgba(255, 99, 132, 0.2)';
        break;
      case 2:
        borderColor = 'rgba(54, 162, 235, 1)';
        backgroundColor = 'rgba(54, 162, 235, 0.2)';
        break;
      case 3:
        borderColor = 'rgba(255, 206, 86, 1)';
        backgroundColor = 'rgba(255, 206, 86, 0.2)';
        break;
    }
    return {
      label: `Evolución - ${mod}`,
      data: data,
      borderColor: borderColor,
      backgroundColor: backgroundColor,
      fill: true
    };
  });
  
  // Dibujar la gráfica de líneas en un solo canvas
  const canvasLine = document.getElementById('evolutionChart');
  if (!canvasLine) {
    console.error('No se encontró el canvas con id "evolutionChart".');
    return;
  }
  const ctxLine = canvasLine.getContext('2d');
  new Chart(ctxLine, {
    type: 'line',
    data: {
      labels: labels,
      datasets: datasets
    },
    options: {
      responsive: true,
      plugins: {
        zoom: {
          zoom: {
            wheel: { enabled: true },
            pinch: { enabled: true },
            mode: 'x'
          },
          pan: { enabled: true, mode: 'x' }
        }
      },
      scales: {
        x: { title: { display: true, text: 'Mes/Año' } },
        y: { title: { display: true, text: 'Total de Aciertos' } }
      }
    }
  });

  // Gráfica de torta: obtener datos globales
  const pieData = await fetchData('get_pieAnaEstDesHabVisChartData.php');
  if (!pieData) {
    console.error("No se pudieron obtener datos para la gráfica de torta.");
    return;
  }
  
  const pieLabels = pieData.map(item => item.modality);
  const pieValues = pieData.map(item => item.total_entries);
  const totalEntries = pieValues.reduce((acc, val) => acc + val, 0);
  
  const canvasPie = document.getElementById('pieChart');
  if (!canvasPie) {
    console.error('No se encontró el canvas con id "pieChart".');
    return;
  }
  const ctxPie = canvasPie.getContext('2d');
  new Chart(ctxPie, {
    type: 'pie',
    data: {
      labels: pieLabels,
      datasets: [{
        data: pieValues,
        backgroundColor: [
          'rgba(75, 192, 192, 0.6)',
          'rgba(255, 99, 132, 0.6)',
          'rgba(54, 162, 235, 0.6)',
          'rgba(255, 206, 86, 0.6)'
        ]
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'top' },
        tooltip: {
          callbacks: {
            label: function(context) {
              const label = context.label || '';
              const value = context.raw;
              const percentage = totalEntries > 0 ? ((value / totalEntries) * 100).toFixed(2) : 0;
              return `${label}: ${percentage}% (${value})`;
            }
          }
        }
      }
    }
  });
}

window.onload = initCharts;
